const char MAIN_page[] PROGMEM = R"=====(
<!DOCTYPE html>
<html lang="ru">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ESP32-C3 Мониторинг - Единая диаграмма</title>

    <!-- Chart.js библиотека для графиков -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

    <style>
        * {
            box-sizing: border-box;
            font-family: 'Segoe UI', Arial, sans-serif;
        }

        body {
            margin: 0;
            padding: 20px;
            background: linear-gradient(135deg, #1a2980, #26d0ce);
            min-height: 100vh;
            color: #333;
        }

        .container {
            max-width: 1400px;
            margin: 0 auto;
            background: rgba(255, 255, 255, 0.95);
            border-radius: 20px;
            padding: 30px;
            box-shadow: 0 15px 35px rgba(0, 0, 0, 0.2);
        }

        .header {
            text-align: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 3px solid #f0f0f0;
        }

        .header h1 {
            color: #1a2980;
            margin: 0;
            font-size: 2.5em;
        }

        .header p {
            color: #666;
            font-size: 1.1em;
        }

        .current-values {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 30px;
        }

        .value-box {
            background: white;
            border-radius: 12px;
            padding: 20px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
            text-align: center;
            border-left: 5px solid;
            transition: transform 0.3s;
        }

        .value-box:hover {
            transform: translateY(-5px);
        }

        .sensor-label {
            font-size: 16px;
            font-weight: 600;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }

        .sensor-value {
            font-size: 32px;
            font-weight: 700;
            margin: 5px 0;
        }

        .unit {
            font-size: 14px;
            color: #777;
            margin-left: 5px;
        }

        /* Цвета для разных датчиков */
        .temp-box { border-left-color: #FF6B6B; }
        .adc1-box { border-left-color: #4ECDC4; }
        .adc2-box { border-left-color: #45B7D1; }
        .adc3-box { border-left-color: #96CEB4; }
        .adc4-box { border-left-color: #FFEAA7; }

        .temp-value { color: #FF6B6B; }
        .adc1-value { color: #4ECDC4; }
        .adc2-value { color: #45B7D1; }
        .adc3-value { color: #96CEB4; }
        .adc4-value { color: #FFEAA7; }

        .main-chart-container {
            background: white;
            border-radius: 15px;
            padding: 25px;
            margin-bottom: 30px;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
            height: 500px;
        }

        .controls {
            display: flex;
            justify-content: center;
            flex-wrap: wrap;
            gap: 15px;
            margin-bottom: 20px;
        }

        button {
            background: linear-gradient(45deg, #1a2980, #26d0ce);
            color: white;
            border: none;
            padding: 12px 25px;
            border-radius: 50px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
            display: flex;
            align-items: center;
            gap: 8px;
            box-shadow: 0 4px 15px rgba(26, 41, 128, 0.3);
        }

        button:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 20px rgba(26, 41, 128, 0.4);
        }

        button:active {
            transform: translateY(0);
        }

        .stop-btn {
            background: linear-gradient(45deg, #FF6B6B, #FF8E8E);
        }

        .clear-btn {
            background: linear-gradient(45deg, #95a5a6, #7f8c8d);
        }

        .status-bar {
            background: rgba(26, 41, 128, 0.05);
            border-radius: 12px;
            padding: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 15px;
        }

        .status-item {
            display: flex;
            flex-direction: column;
            align-items: center;
            min-width: 150px;
        }

        .status-label {
            font-size: 14px;
            color: #666;
            margin-bottom: 5px;
        }

        .status-value {
            font-size: 20px;
            font-weight: 700;
        }

        .green { color: #2ecc71; }
        .orange { color: #e67e22; }
        .red { color: #e74c3c; }

        .legend {
            display: flex;
            justify-content: center;
            flex-wrap: wrap;
            gap: 15px;
            margin-top: 20px;
        }

        .legend-item {
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .legend-color {
            width: 20px;
            height: 20px;
            border-radius: 4px;
        }

        @media (max-width: 768px) {
            .container {
                padding: 15px;
            }

            .header h1 {
                font-size: 1.8em;
            }

            .main-chart-container {
                height: 400px;
                padding: 15px;
            }

            .controls {
                flex-direction: column;
                align-items: center;
            }

            button {
                width: 100%;
                justify-content: center;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>📈 ESP32-C3 Мониторинг - Единая диаграмма</h1>
            <p>Все данные отображаются на одной диаграмме. Обновление каждые <span id="interval">5</span> секунд</p>
        </div>

        <!-- Текущие значения -->
        <div class="current-values">
            <div class="value-box temp-box">
                <div class="sensor-label">🌡️ Температура</div>
                <div class="sensor-value temp-value" id="tempValue">--</div>
                <div class="unit">°C</div>
            </div>

            <div class="value-box adc1-box">
                <div class="sensor-label">🔌 ADC1 (GPIO0)</div>
                <div class="sensor-value adc1-value" id="adc1Value">--</div>
                <div class="unit">мВ</div>
            </div>

            <div class="value-box adc2-box">
                <div class="sensor-label">⚡ ADC2 (GPIO1)</div>
                <div class="sensor-value adc2-value" id="adc2Value">--</div>
                <div class="unit">мВ</div>
            </div>

            <div class="value-box adc3-box">
                <div class="sensor-label">💡 ADC3 (GPIO2)</div>
                <div class="sensor-value adc3-value" id="adc3Value">--</div>
                <div class="unit">мВ</div>
            </div>

            <div class="value-box adc4-box">
                <div class="sensor-label">📊 ADC4 (GPIO3)</div>
                <div class="sensor-value adc4-value" id="adc4Value">--</div>
                <div class="unit">мВ</div>
            </div>
        </div>

        <!-- Единая диаграмма -->
        <div class="main-chart-container">
            <canvas id="mainChart"></canvas>
        </div>

        <!-- Легенда для диаграммы -->
        <div class="legend">
            <div class="legend-item">
                <div class="legend-color" style="background: #FF6B6B;"></div>
                <span>Температура (°C)</span>
            </div>
            <div class="legend-item">
                <div class="legend-color" style="background: #4ECDC4;"></div>
                <span>ADC1 (мВ)</span>
            </div>
            <div class="legend-item">
                <div class="legend-color" style="background: #45B7D1;"></div>
                <span>ADC2 (мВ)</span>
            </div>
            <div class="legend-item">
                <div class="legend-color" style="background: #96CEB4;"></div>
                <span>ADC3 (мВ)</span>
            </div>
            <div class="legend-item">
                <div class="legend-color" style="background: #FFEAA7;"></div>
                <span>ADC4 (мВ)</span>
            </div>
        </div>

        <!-- Управление -->
        <div class="controls">
            <button onclick="startAutoUpdate()">
                <span>▶️</span> Автообновление
            </button>
            <button class="stop-btn" onclick="stopAutoUpdate()">
                <span>⏸️</span> Пауза
            </button>
            <button onclick="updateAllData()">
                <span>🔄</span> Обновить сейчас
            </button>
            <button class="clear-btn" onclick="clearChart()">
                <span>🗑️</span> Очистить график
            </button>
            <button onclick="toggleLegend()">
                <span>📋</span> Легенда
            </button>
        </div>

        <!-- Статус бар -->
        <div class="status-bar">
            <div class="status-item">
                <div class="status-label">Последнее обновление</div>
                <div class="status-value" id="lastUpdate">--:--:--</div>
            </div>

            <div class="status-item">
                <div class="status-label">Следующее обновление через</div>
                <div class="status-value">
                    <span id="countdown">5</span> сек
                </div>
            </div>

            <div class="status-item">
                <div class="status-label">Статус</div>
                <div class="status-value green" id="status">Активен</div>
            </div>

            <div class="status-item">
                <div class="status-label">Точек на графике</div>
                <div class="status-value" id="dataPoints">0</div>
            </div>
        </div>
    </div>

    <script>
        // Конфигурация
        const UPDATE_INTERVAL = 5000; // 5 секунд
        const MAX_DATA_POINTS = 30;   // Максимум точек на графике

        // Глобальные переменные
        let updateTimer = null;
        let countdownTimer = null;
        let countdownValue = UPDATE_INTERVAL / 1000;
        let chartVisible = true;

        // Цвета для линий графика
        const chartColors = [
            '#FF6B6B', // Температура - красный
            '#4ECDC4', // ADC1 - бирюзовый
            '#45B7D1', // ADC2 - голубой
            '#96CEB4', // ADC3 - зеленый
            '#FFEAA7'  // ADC4 - желтый
        ];

        // Данные для графика
        const chartData = {
            labels: [], // Временные метки
            datasets: [
                {
                    label: 'Температура (°C)',
                    data: [],
                    borderColor: chartColors[0],
                    backgroundColor: chartColors[0] + '20',
                    borderWidth: 3,
                    tension: 0.4,
                    yAxisID: 'y',
                    pointRadius: 3,
                    pointHoverRadius: 6
                },
                {
                    label: 'ADC1 (мВ)',
                    data: [],
                    borderColor: chartColors[1],
                    backgroundColor: chartColors[1] + '20',
                    borderWidth: 2,
                    tension: 0.4,
                    yAxisID: 'y1',
                    pointRadius: 2,
                    pointHoverRadius: 5
                },
                {
                    label: 'ADC2 (мВ)',
                    data: [],
                    borderColor: chartColors[2],
                    backgroundColor: chartColors[2] + '20',
                    borderWidth: 2,
                    tension: 0.4,
                    yAxisID: 'y1',
                    pointRadius: 2,
                    pointHoverRadius: 5
                },
                {
                    label: 'ADC3 (мВ)',
                    data: [],
                    borderColor: chartColors[3],
                    backgroundColor: chartColors[3] + '20',
                    borderWidth: 2,
                    tension: 0.4,
                    yAxisID: 'y1',
                    pointRadius: 2,
                    pointHoverRadius: 5
                },
                {
                    label: 'ADC4 (мВ)',
                    data: [],
                    borderColor: chartColors[4],
                    backgroundColor: chartColors[4] + '20',
                    borderWidth: 2,
                    tension: 0.4,
                    yAxisID: 'y1',
                    pointRadius: 2,
                    pointHoverRadius: 5
                }
            ]
        };

        // Настройки графика
        const chartConfig = {
            type: 'line',
            data: chartData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                interaction: {
                    mode: 'index',
                    intersect: false
                },
                stacked: false,
                plugins: {
                    legend: {
                        display: true,
                        position: 'top',
                        labels: {
                            usePointStyle: true,
                            pointStyle: 'circle',
                            padding: 20,
                            font: {
                                size: 12
                            }
                        }
                    },
                    tooltip: {
                        mode: 'index',
                        intersect: false,
                        callbacks: {
                            label: function(context) {
                                let label = context.dataset.label || '';
                                if (label) {
                                    label += ': ';
                                }
                                label += context.parsed.y.toFixed(1);
                                if (context.datasetIndex === 0) {
                                    label += '°C';
                                } else {
                                    label += ' мВ';
                                }
                                return label;
                            }
                        }
                    }
                },
                scales: {
                    x: {
                        grid: {
                            color: 'rgba(0,0,0,0.05)'
                        },
                        ticks: {
                            maxTicksLimit: 10,
                            font: {
                                size: 11
                            }
                        },
                        title: {
                            display: true,
                            text: 'Время',
                            font: {
                                size: 14,
                                weight: 'bold'
                            }
                        }
                    },
                    y: {
                        type: 'linear',
                        display: true,
                        position: 'left',
                        title: {
                            display: true,
                            text: 'Температура (°C)',
                            font: {
                                size: 14,
                                weight: 'bold'
                            }
                        },
                        grid: {
                            drawOnChartArea: false
                        },
                        ticks: {
                            font: {
                                size: 11
                            }
                        }
                    },
                    y1: {
                        type: 'linear',
                        display: true,
                        position: 'right',
                        title: {
                            display: true,
                            text: 'Напряжение (мВ)',
                            font: {
                                size: 14,
                                weight: 'bold'
                            }
                        },
                        grid: {
                            drawOnChartArea: false
                        },
                        ticks: {
                            font: {
                                size: 11
                            }
                        }
                    }
                }
            }
        };

        // Инициализация графика
        let mainChart = null;

        // Функция инициализации графика
        function initChart() {
            const ctx = document.getElementById('mainChart').getContext('2d');
            mainChart = new Chart(ctx, chartConfig);
        }

        // Функция для получения всех данных
        function updateAllData() {
            fetch('/getAllData')
                .then(response => {
                    if (!response.ok) throw new Error('Ошибка сети');
                    return response.json();
                })
                .then(data => {
                    // Обновление текущих значений
                    document.getElementById('tempValue').textContent = data.temperature.toFixed(1);
                    document.getElementById('adc1Value').textContent = data.adc1;
                    document.getElementById('adc2Value').textContent = data.adc2;
                    document.getElementById('adc3Value').textContent = data.adc3;
                    document.getElementById('adc4Value').textContent = data.adc4;

                    // Добавление времени
                    const now = new Date();
                    const timeString = now.toLocaleTimeString();

                    // Добавление данных в график
                    chartData.labels.push(timeString);
                    chartData.datasets[0].data.push(data.temperature);
                    chartData.datasets[1].data.push(data.adc1);
                    chartData.datasets[2].data.push(data.adc2);
                    chartData.datasets[3].data.push(data.adc3);
                    chartData.datasets[4].data.push(data.adc4);

                    // Ограничение количества точек
                    if (chartData.labels.length > MAX_DATA_POINTS) {
                        chartData.labels.shift();
                        chartData.datasets.forEach(dataset => dataset.data.shift());
                    }

                    // Обновление графика
                    if (mainChart) {
                        mainChart.update();
                    }

                    // Обновление времени последнего обновления
                    document.getElementById('lastUpdate').textContent = timeString;

                    // Обновление счетчика точек
                    document.getElementById('dataPoints').textContent = chartData.labels.length;

                    console.log('Данные обновлены:', data);
                })
                .catch(error => {
                    console.error('Ошибка получения данных:', error);
                    document.getElementById('status').textContent = 'Ошибка';
                    document.getElementById('status').className = 'red';
                });
        }

        // Функция автообновления
        function startAutoUpdate() {
            if (updateTimer) clearInterval(updateTimer);
            if (countdownTimer) clearInterval(countdownTimer);

            updateTimer = setInterval(updateAllData, UPDATE_INTERVAL);
            startCountdown();

            document.getElementById('status').textContent = 'Активен';
            document.getElementById('status').className = 'green';

            // Сразу обновляем данные
            updateAllData();
        }

        // Функция остановки автообновления
        function stopAutoUpdate() {
            if (updateTimer) {
                clearInterval(updateTimer);
                updateTimer = null;
            }
            if (countdownTimer) {
                clearInterval(countdownTimer);
                countdownTimer = null;
            }

            document.getElementById('status').textContent = 'На паузе';
            document.getElementById('status').className = 'orange';
            document.getElementById('countdown').textContent = '--';
        }

        // Таймер обратного отсчета
        function startCountdown() {
            countdownValue = UPDATE_INTERVAL / 1000;
            document.getElementById('countdown').textContent = countdownValue;

            countdownTimer = setInterval(() => {
                countdownValue--;
                document.getElementById('countdown').textContent = countdownValue;

                if (countdownValue <= 0) {
                    countdownValue = UPDATE_INTERVAL / 1000;
                }
            }, 1000);
        }

        // Очистка графика
        function clearChart() {
            chartData.labels = [];
            chartData.datasets.forEach(dataset => dataset.data = []);

            if (mainChart) {
                mainChart.update();
            }

            // Сброс текущих значений
            document.getElementById('tempValue').textContent = '--';
            document.getElementById('adc1Value').textContent = '--';
            document.getElementById('adc2Value').textContent = '--';
            document.getElementById('adc3Value').textContent = '--';
            document.getElementById('adc4Value').textContent = '--';
            document.getElementById('dataPoints').textContent = '0';

            console.log('График очищен');
        }

        // Переключение легенды
        function toggleLegend() {
            if (mainChart) {
                const legend = mainChart.options.plugins.legend;
                legend.display = !legend.display;
                mainChart.update();
            }
        }

        // Инициализация при загрузке страницы
        document.addEventListener('DOMContentLoaded', function() {
            initChart();
            startAutoUpdate();
        });
    </script>
</body>
</html>
)=====";